/*
 *  mixmax.c
 *  A Pseudo-Random Number Generator
 *
 *  Created by Kosta on Sun Feb 22 2004.
 *  Will be released with some Open Source Licence soon. In the meantime, 
 *  permission is given for non-commercial or commercial use and modification, but not redistribution.
 *
 *	Generator described in 
 *	N.Z.Akopov,G.K.Savvidy and N.G.Ter-Arutyunian, Matrix Generator of Pseudorandom Numbers, 
 *	J.Comput.Phys.97, 573 (1991); 
 *	Preprint EPI-867(18)-86, Yerevan Jun.1986;
 */


#include <stdio.h>
#include <stdlib.h>

#include "mixmax2008.h"  // stdint.h is included there

myuint* rng_alloc()
{
	myuint  *p = malloc((N+1)*sizeof(myuint));
	return p;
}

int rng_free(myuint *X)
{
	free(X);
	return 0;
}

int seedvector(myuint *X, unsigned int seed)	
{
int i;
	for (i=0; i <= N; i++){
			X[i] = ZERO;
	}
	if (seed<=N && seed>=1){
	X[seed] = SEEDVAL;
	}else{
		fprintf(stderr, "Out of bounds seed index, is not (seed<=n and seed>=1)\n"); return -1;
	}
	//X[0] = N;
	return 0;
}

int seed_lcg(myuint *X)	/* an LCG generator is used to seed */
{
	int i;

	X[1] = SEEDVAL;
	for (i=2; i <= N; i++){
		X[i] = MOD_MERSENNE(SUFF(4563402761) * X[i-1]) ; //& 0xffffffff
	}
	X[0] = N;
	return 0;
}

inline void iterate(myuint *X){
	int i;
	myuint partial[N+1];  /* the array for the partial sums vector  */
	myuint temp;

	partial[0] = ZERO;	
	X[0] = ZERO;
	for (i=1; i<=N; i++){
		partial[i] = MOD_MERSENNE(partial[i-1] + X[N-(i)]);
	}
	X[N] = MOD_MERSENNE(X[N]+partial[N]);
	temp = X[N-1];
	for (i=1; i<N; i++){
		X[N-i] = MOD_MERSENNE(X[N-(i-1)] + partial[i]);
	}
	X[N-2] = MOD_MERSENNE(X[N-2] - temp + MERSBASE); /* the special entry of the matrix taken care here! */
}

#ifdef IT_UNR
inline void iterate_unrolled(myuint *X){
	int i;
	myuint partial[N+1];  /* the array for the intermediate vector  */
	myuint temp;
	//myuint _temp;
	
	partial[0] = ZERO;	
	
#define DO_NOW(i) partial[i]=MOD_MERSENNE(partial[i-1]+X[N-(i)])
	
#define X0(n) DO_NOW(n);
	
#define X2(n) X0(n) X0(n+1)
#define X4(n) X2(n) X2(n+2)       //for N=4
#define X8(n) X4(n) X4(n+4)
#define X16(n) X8(n) \
X8(n+8)
#define X32(n) X16(n) X16(n+16)    //for N=32
#define X64(n) X32(n) X32(n+32)
#define X128(n) X64(n) X64(n+64)
#define X256(n) X128(n) X128(n+128)   //for N=256

	ALL(1)
	X[N] = MOD_MERSENNE(X[N]+partial[N]);
	temp = X[N-1];
	
#undef DO_NOW
#define DO_NOW(i) X[N-(i)] = MOD_MERSENNE(X[N-(i-1)] + partial[i])
	
	ALL(1)
	X[0] = ZERO;   
	X[N-2] = MOD_MERSENNE(X[N-2] - temp + MERSBASE); /* the special entry of the matrix taken care here! */
}
#endif

myuint get_next(myuint *X){
	int i;
	i=X[0];
	if (i>=N){
		i=1;
		IT(X);
	    } else {
		i++;
	}
	X[0]=i;
	return X[i];
}

long double get_next_float(myuint *X){
	int i;
	i=X[0];
	if (i>=N){
		i=1;
		IT(X);
	    } else {
		i++;
	}
	X[0]=i;
	return ((long double)X[i] * INV_MERSBASE);
}

#if __LP64__
#if (N==16 || N==32)
myuint *skip_1M(myuint *X){
	/*
	 * Skips 2^20 = 1048576 steps, by applying a precomputed matrix to X
	 */
#if (N==16)
	__uint128_t M1M[N][N] =  // this is to store the transpose of the matrix directly, as a list of lists
 {
	{401919016885129715,	1778761845647486798,	151945339421938638,	217933623014157965,	595620117970336451,	600186237365314687,	1534892074714932803,	866823534232771807,	1830867485065800292,	1577599723013837070,	1504630541826524856,	654806591792883329,	164679981183892192,	184491700436836821,	163207562989798338,	2071379129780234499},
	{1218970042124108194,	311054972228670015,	918007253346421607,	670290880441282880,	876938210096667963,	1909063172740378211,	2292366509555780091,	975635141150636705,	1090108725484845077,	224862620408216103,	744402184099674857,	688059414031841957,	1817633956026202619,	122111706289815226,	205775837883875304,	560879005413056128},
	{944453686827205184,	1894074805781327427,	1752128544199568181,	49222971161895228,	720219784266579831,	317791430664751867,	1783169568320198210,	348399028524954685,	2258643131080669893,	1126996504879144232,	1904643043854398602,	325474551478814166,	666775276584803474,	800850717533965002,	59731712142793631,	248344112777952270},
	{810851551039222136,	875042330896373724,	1080147330262445274,	989043030404599660,	1697679234970178516,	1661360398844155505,	453376430038125360,	2180233171487125197,	780650706675940114,	1840138497452607563,	253099458228720417,	782911536321062424,	806782178246696452,	84772525191203709,	1479589728778114778,	2219530595615405909},
	{497954040810938110,	918922980785466464,	1160596546319693840,	159340227414273708,	2216357347716919000,	198008596572839957,	26721516685596664,	1660199684775022087,	1890973068580297713,	2075721831924993668,	237099363151389099,	226761245490401255,	1851966618377760278,	1382652109729815481,	1674537342062136367,	593604736199741974},
	{779433592424822526,	1811754715640744487,	1132609891485140939,	1302416152295722674,	1363786542214502942,	1092092650791364177,	1754992850332762821,	163642887936613331,	727219172540158694,	314272147017398411,	2238934881977529412,	2166564597557679051,	1806186917827606370,	824873132859509406,	374688685054733302,	1869484955346157956},
	{1408162477689761320,	568640312410881333,	76367480783055724,	2059834529497476840,	709202813130671691,	1278756558297865693,	2046297379883644502,	2106418054757400607,	729571925534872236,	486214750757845946,	1408964225015672217,	1959262978707847329,	1983598565825718884,	974418063298301778,	267094155989203331,	1380683037261024188},
	{2008603179471334524,	1116722060295444104,	196552757636188094,	1067579606091358009,	809191132888051558,	1106972368290566061,	2159325652534291682,	547387830050522204,	1063016759880212176,	382683307395930420,	33441936262275062,	1714569527239950594,	544302477212392215,	678023236967041984,	1123962993737094150,	159499626923673360},
	{1832973178242566649,	588813952442550166,	874427032845886555,	84050257511766235,	2157205489747905957,	1933243287596878785,	651464113334632194,	367740048385997959,	768969180800751007,	243737747044164262,	1396856909076470359,	208718944905231873,	1759518242301585839,	1342520820466626647,	381628410635782190,	1980831831484984969},
	{678756841173909508,	1488278823555000832,	1319214470096202890,	271680086389055601,	1296901873936220137,	1187439829726571862,	1175502822059077319,	1897334446428657080,	39380300294953552,	2196511343082127341,	1612258075229270893,	813880954552449731,	1173333036427927757,	1276073133853601282,	2007018403966211310,	1945136836748164181},
	{1631913989182202494,	2028315156384553052,	2226246799450159786,	2064749904232868074,	59654891905522972,	1965842877758394928,	1189244488178561588,	1936852604500858237,	2174163678218194030,	921398511965071859,	227162594927293670,	916726269161884822,	5793296941698217,	1015326579739793358,	1209625447240575917,	1326565388082946479},
	{1826016025995768951,	1118022086578040174,	1956816045179278819,	1345280601144452366,	1863125082698482001,	2174273452533977406,	914452976765741103,	1390503036954958163,	1578019919611870469,	1179669076650681544,	356234106373622531,	1066182531899228685,	984454939959446972,	606990761036529121,	754580025625985434,	412232490514940524},
	{1880220483861808662,	76477881701180142,	438680709806718827,	669632402841910358,	1957722523989116815,	983790740708503021,	606829217139258963,	1136797416182651993,	1633933367012383368,	117870255402014696,	1559707359012176435,	1681475076688061641,	571801526342477939,	28843919726513397,	198654942333264884,	299534604011394951},
	{369056457722011203,	182219740214161825,	735857547914169228,	130938409790324395,	1341331445146489955,	382835676539917609,	1019194397013777540,	2245080765144526444,	1885113063562582114,	581291322501124454,	1778578839803046592,	2195243236239051385,	1627305860827372431,	1149876880801896926,	1756540087630191624,	1948572868254238285},
	{2266091925329307516,	430216943578614804,	437172971569178694,	1733670407801055595,	1943775593846808003,	2126417966992139668,	511064038333460616,	1829775111952478008,	642136577303966665,	1598741019449407007,	1256455527935719001,	687575545405286445,	698942007582666345,	1522076208196732172,	571730039491881202,	1200615004337471074},
	{1099735342659341007,	85957055829719311,	2146090137271673430,	591053998575358215,	1971323409229390522,	2202960615197093799,	2208622592613437273,	2084135247117763514,	1650568904201149284,	48611482646472432,	1144933202401874466,	2053039828721383724,	1001118722730390884,	163207562989798338,	2071379129780234499,	929000180451336868}
 };
#elif (N==32)
	__uint128_t M1M[N][N] =
 {
	{1033320363213560109,	759148076592861683,	373303505158416466,	1845930249606328125,	1619093666616994384,	909553320460054711,	1194828727775862185,	328153852856768463,	178501366542456123,	588957861199258691,	919710041375453063,	372663959257305618,	2173085844869826325,	677386534129574470,	695385176356644220,	1816946458485238953,	2150343115993129331,	370351723586118534,	863018726878868257,	1212101098395016756,	2230170824560250490,	1451290767644150872,	1803735344394424056,	901079320918778624,	633545121917813182,	1939913021650214451,	1826366147867103201,	1950912653088462930,	926958741017377556,	1847418684129335487,	1305342225415306363,	1822120710717664322},
	{216821905227106245,	1458438788839863929,	1995229499279914540,	1372839606084610521,	1765996025234356025,	2168765564029989857,	1723621707970238183,	1383797368501628900,	855743620622614233,	823224407417345095,	1547651762772842263,	1078300917427073701,	1195266834961720138,	702304888591519318,	23668988559584657,	572231812301763812,	1237243072867587759,	465084727951143789,	1118853962636785629,	2010940833464053948,	184032576186460453,	952913837059237502,	850175225105545663,	1504160282792859070,	569463396716614094,	969788382994776804,	420103408373558994,	968043425306126509,	1258221120037594920,	2148648832803013259,	83652133629670660,	788563740112948404},
	{1680540965984689084,	1925534196476456598,	2144901936498566864,	2132074371910267553,	646993250419122492,	394237538275352067,	1181419701912680419,	1240704605439129786,	228994522713645391,	324391170494495396,	1421639475499912400,	1785787997364393301,	1395982510558555139,	1008027145520268494,	1798606465986923037,	564146679706687957,	1776340558650850820,	1919320941969719018,	1208655844983697898,	1857493957651226752,	34455707694056552,	1870816865282409866,	992053384324761414,	1529569862175198696,	856681609320036171,	1911964983186826777,	1012705285644544597,	2025746826949750037,	425966966592097385,	1890528814455136510,	144035972262997080,	1167805051057728908},
	{2073903251003122918,	526856889000613506,	215296760472649040,	343806926045942764,	698882487588360291,	812298846654489948,	2096607217705090937,	961818904462079695,	1284420113069036735,	1471228383761442798,	1034368333504515096,	2243178733437384495,	884887269434092502,	1582342470780166054,	1567986755404941968,	1822224108360866266,	70904273307471115,	1891590008717024079,	1467741972999934277,	1867769799225268258,	131238861906232398,	864273651187759191,	1843668463173865731,	898617032842149232,	663302522140321299,	2077468611936798227,	705194392736444255,	862090211307705711,	118873259699881222,	337993324022840520,	1632408796107259761,	204419810896323500},
	{1794910477053969054,	248032897105357011,	334643267402851221,	1541106953280932028,	184836303687744938,	1814133127193871901,	571651520416451761,	2218785895012969644,	1425777056528437833,	2168620909571846062,	1997396115719264045,	1376726634410137890,	2106356077311157988,	1512142685111940287,	344650332700556758,	838485878574743853,	339604712326526807,	2210593906065396941,	1444033805739314570,	1024208536014300000,	321588971012429344,	1070975310439242875,	1913928604760077506,	1664283763213352545,	633283984403735042,	998903765298370316,	1527884095636134205,	812795950585498733,	1227908401342653553,	175177965739541780,	1091300842804238481,	814938610737828491},
	{589502569251695447,	1469888167886055916,	1423858632667296642,	572665350910798817,	691340927488544232,	950117883674861123,	2229445647562592661,	1642923887391931468,	1933308942442356400,	390864999320853207,	1520439007867561434,	174864580352873542,	2000991281199121992,	1185865017801658748,	1349748159628787973,	1257553419813861426,	1087949920417857353,	15309182833399315,	1728156223534910082,	926714643656050361,	433103648833405859,	685560024863479976,	859180969608333662,	440684969525940765,	1203024201919286200,	2281940919467710493,	1654773383054578250,	1083364838235520144,	2121998887148910084,	1712483357265808623,	12362607456243040,	550192889501217201},
	{1054135315111525499,	768688146846919588,	2153281632244417336,	1614819825822079851,	716565653712417639,	1828907558884169672,	154126557265301867,	222150474374753389,	1168747192468161119,	1270432505165428946,	100024181974582925,	1586939274879305498,	2130396649742752173,	676104754762507250,	1762426033651622455,	359421707598173750,	669840437853022263,	1968902923866759478,	2101510719224331057,	1771310960419872523,	1530343279509239452,	2231465807720678182,	1612470279573280039,	618952810645149050,	1433472286833708282,	1344015388615193628,	915266357795549898,	571397651167743077,	1666975641076157810,	2275339198010334100,	943945739578381515,	1239267381321941550},
	{2006477314548765781,	78135441406099882,	943345831054538164,	1819203536301293068,	755702440781048106,	1160273590042429278,	1578756800148567510,	449816366203656799,	514982256225608574,	2228587451093641838,	1747146450479623887,	1344292889949243089,	1703133414603187811,	407252859800638168,	860546710528587933,	1554748929333820114,	934393950993183291,	101654897390186557,	2257421848705879892,	592347032511640143,	1015671859323222205,	1721279450676443676,	814975991167814307,	158423400971446177,	256930221618359889,	2022482800686056394,	75202220410288473,	2050157592271594134,	1618712842792972771,	2215329054431639945,	532352029541165626,	1875528871700519990},
	{67769129905191590,	2167664786253481062,	1376510258520956061,	1059594771862937995,	79936504647523047,	442128771728233760,	89418315606788947,	1993040019226582961,	982692995421332769,	1287903732810105329,	408747136423460678,	2242437894252454528,	1090607069520244207,	1079567634021307734,	883265088578992019,	1628129212319214952,	991748028019182542,	1539262914182592557,	1879578364231840795,	913063839122680052,	1718637709271625328,	1718049304726148942,	629255161101398975,	47417228570588780,	77352754378828985,	456811517459914666,	1833234878516817836,	965519544276716512,	390057996314712025,	2134166895546064659,	2155318910852945790,	120758319503949737},
	{2113327919132281798,	1441497142611208363,	142247321533411425,	638986530393348510,	1774141384013017013,	942453751387757153,	174893925047822367,	625803896681606492,	345684953084508649,	1766792252346896077,	2083180770076735190,	1513298516413207609,	1650406201245423798,	160229503935181629,	2148221292646756606,	238717167837841961,	1877293878259981768,	2053078261215353746,	751211408171831748,	831394894582358890,	161054909406916082,	1394172182293806154,	366740805367583857,	1324505461173007457,	2205476591311031271,	2168699117238637208,	2133779938124282384,	764733623587403928,	244042719765154608,	26641989539012956,	2053004736660489373,	1472442782951032003},
	{516986073043871987,	2304544845073145620,	450202981786919234,	185076464604902804,	1462168544663938819,	400115791190424662,	1943571337429726247,	138520934598477997,	16672334708100207,	774776895279570391,	1840876879811456253,	1436072987739225681,	1420374580962135611,	2056590023350310532,	1902974713730017675,	2259215044397285875,	1105252592080050557,	1055357051589938193,	1653717383892789343,	740541159537463751,	1858237866963104801,	660906774670477923,	412498430585055059,	2060544810522340703,	1711020162197218716,	820446630619011298,	1918978851422176941,	2263213995040688083,	2092102992349629234,	734302733096156974,	224960092745655204,	1950690562468032956},
	{1108207876991818028,	985112544220478070,	743737852271800013,	625726166222211100,	493293690403995157,	1255420981991533216,	1360646258668205807,	557671677129625492,	2014780622875008759,	2072480264020974065,	2151457802496562414,	2214311057567186172,	524206116374501077,	1995407782897452352,	494629595902363329,	1647994667466675125,	1638261200120071506,	1302652488497107174,	2272093671094095002,	2180708491009973413,	1805193622406199888,	654856811106137213,	60135806906674857,	192630246766200022,	2232847446943567734,	1669110734015461519,	2121657734947516877,	159435307468260464,	920535859430143381,	1951858352184916420,	1441963476653300992,	702758458044514986},
	{757983051751410159,	1303156698130413143,	906883715321062217,	2148087187274313914,	953058642578253085,	839741417316986774,	2155063998489922053,	2199401824187202336,	1315239459254025354,	1653656498847277721,	1600267095866332082,	2136193568183980376,	1810348411186292177,	628147694977437691,	1628257738168008839,	1322289933891423506,	1252452380494444078,	1468130470452823781,	764991132745555107,	2285877349300528010,	1681284740305655359,	2047536667342521594,	37080947622054262,	1731198056916005972,	1448779921390023338,	2267001150556293384,	1779327569422670016,	1040880301433428292,	2063266993607693722,	948744238179437611,	863570962059981915,	353123851347252829},
	{726382597075710781,	722893352425720109,	1038028478340437928,	454856104640527640,	2145525640124424810,	620603071922432892,	2134225558306048722,	1538546254755963161,	127610771200288023,	1025775589583184768,	1435852883533327587,	1636873033393073797,	2224669677550045844,	85420267085623157,	1686035846600550939,	1805341076929984897,	848796273426315445,	701007346536459003,	535371132204101356,	1742976821213356275,	1922222038816003387,	1453397870040924167,	1349419881876737504,	1564466249388262984,	809063401225398350,	654203944598678729,	643695493360878516,	728342428838325734,	19687453778828313,	762440944665458493,	2251473133387652753,	285178447466662838},
	{544731871956160801,	327498828370087954,	2266058523553858456,	2083644031172995886,	692271811772484182,	317075592324533047,	1907844565405857461,	1108381395070141961,	103375672018108531,	1634903667925187027,	2023982243490129411,	1344226797447473056,	1417014923694623020,	1696477348787078822,	753963753660358133,	1965234668126907643,	2186655381220722247,	2084918542472898206,	1418853909373915649,	597980586914043122,	890210042165438639,	580343791204692738,	1303371254296728012,	1747483435854129284,	833208055576837822,	268681669293155648,	368016457525537934,	1694626176410011794,	1575345926307242621,	1186438931507718380,	576137651151479375,	1333532295501629640},
	{1003293943410227728,	927250809087325237,	1491420471338039653,	744893629147929556,	301541698742274079,	1502337458206471928,	1459813923778706351,	600236725617085549,	1824852887942198849,	665646231286382986,	1252151889736836933,	260272033206484580,	214275472179731876,	1682813392071285388,	450423605089676629,	2204998798118206673,	2282262926302554068,	133554584067583796,	1354280697547426445,	1545574665899209432,	788812645254688922,	2124640937894647875,	2085240821060727908,	2157348920539300556,	751532205148188641,	1878406797190075649,	593617875238180053,	1304076356488727539,	209972282073563775,	1966928918371839279,	1610436918349978267,	1206645178128999948},
	{2078318112248030840,	998560051491660584,	535264088119026513,	1849368395647184457,	676111362958454113,	1946079498794045000,	1675504402662332073,	663409481236253337,	315035680320951433,	537979304477873729,	2141314523516846193,	2280397243172419490,	1774629941147600574,	2067099604116011093,	510977169139793552,	1658647912137789101,	864569358542889126,	1590218798189966500,	802567124995547111,	1117100615115130885,	944493744509324529,	1701661753347967267,	410065249138301378,	1762878432119387646,	658646476414047488,	1055921556817317099,	1441986758915094738,	1701684601956887777,	278306901749259431,	1846247556457286510,	441575896022266227,	338893176334783208},
	{974822046445804061,	1503680871905237593,	2159862471938335185,	944012640766980488,	3074567420253652,	1663991991121455848,	2277302955068277549,	2164784370754937026,	132530850750222306,	1844322473858717438,	271740456590342957,	133168535511321442,	1847545932153175333,	1700412420381690217,	65840430647911162,	2291981063562473428,	1706588384721342430,	2300879743285313331,	543197677360014576,	345355351621373899,	439857150799080021,	191087737267107704,	1587843639983949223,	1897004711166611787,	447433726689960824,	1832302518894140863,	793001280894681440,	1352566245344804331,	414776848097599239,	889095546349358811,	1725566194542733741,	1578557882908248138},
	{127696506137308397,	1921159935082983231,	1665150880308695229,	1321500068298575856,	1164382290053201806,	2290107694219627093,	638491598129813093,	2200921722024453441,	1522741936574683857,	1685167622089440950,	1114466213785694770,	1384109793545522344,	2060210188213511016,	1522794530149410736,	1125127137140071380,	916529037632773419,	2212354383109518066,	1499394551231878377,	450181691563961020,	194665703960955829,	105503178197429617,	637396268267242321,	1339571979445649129,	1692787228491356166,	619952426153115264,	1514616871306088551,	487169756432608991,	671499041107947306,	1198982279341626470,	2166286617558608247,	2237786545455125063,	703713483849507304},
	{514753228477276603,	685884143686510133,	2091197321544207533,	1843755644827686197,	211228991762623170,	1029313496908769067,	2038779990345082125,	1161336583372365912,	1017149916339689236,	2103083768599102567,	1293214514799059726,	1060416259277954990,	163640547574863397,	2303916061997391450,	29114670507242250,	1018895710311047196,	2159220586203556745,	165678142276652178,	2224921921482998556,	1446091474573859738,	229174234166118061,	414372412031698363,	1057595618467617513,	1031486437667858088,	971672556428367141,	454775367040633737,	117079608464997770,	1343071002425653252,	198960763667076011,	2019861907276414394,	1137634679554163732,	444163887153822434},
	{783981118376266855,	599366827918129291,	296923705019013773,	1143239057747990511,	2075077361929955136,	1784143502503113734,	2004368302320488983,	1934374295829372974,	429680209098394791,	737608973476511199,	963660050848327783,	1712163195801585333,	1952363575790745984,	862161417357410381,	1151408918564911033,	1523623252498143853,	1808301291908180635,	189497191613346346,	578750603142791219,	19696463750462508,	651416794526276826,	911159353079483956,	1469473416024161407,	441755267038474458,	107891276587363019,	2151439915544059979,	673644135170806989,	1952354407566435055,	1851876345421789610,	2158998352972895833,	1873437196994220541,	37482813653202401},
	{1841663961644891494,	2210862221492309126,	1062672425495667194,	1612495338841585939,	110654570761963963,	2084429680069327052,	229933628041367002,	226736757198969554,	789359909315737218,	1177150819558788351,	206248128046782405,	1040362077739644979,	1446618568581694118,	529327004838172560,	1929253424097104451,	2132257066263085639,	1990810965903591462,	1953271137121942414,	143317267417386352,	73318293884732190,	1214057302966880498,	1528970143793001767,	1984280932699567943,	254772562058172386,	934036919749333422,	2016049743089576016,	176303471005405233,	1116188496390712147,	1002168330904808086,	2059646774080993343,	2298134798669377272,	303396705220583399},
	{520030519163019135,	2256562931588748689,	1613779058263981245,	1711043510675167480,	1593665875378014345,	1105580130820847323,	1759085472834027905,	1133455088032515440,	1505568213089091908,	2186194198395923731,	917505778045073866,	218318890449082229,	987801575652927431,	1607042191628150237,	2273210821883658222,	2044086042234687754,	1381074154840077639,	21257429652127853,	175037228618947368,	1244715393034744156,	475703845163812876,	824467566308623880,	2178080330423457541,	2209379726340395504,	1919671003305816529,	1465709140320068793,	1171235717728833761,	461811864822575469,	1244660922805147461,	665993456337054882,	1960295195189090853,	416989391130840052},
	{564306614453769158,	33724537412672453,	869051998144198130,	1741683363816901959,	1318481036601508622,	827370390694268223,	883906610608165670,	1641082639643963746,	2299770970874932635,	253525513670109993,	591117519691467339,	292840471363569518,	1262760523897467660,	823402335422721890,	1117820088121283036,	1856725807038132109,	1096283394406052225,	236651770605814581,	2007809206648759297,	1314688672687817400,	116777146341432465,	123284809711802759,	509219232062860264,	898473475301474535,	1730910231669295882,	1603676581929667093,	302048076997034550,	1994570625657201481,	1822973286862140673,	1576940775158142666,	1578183147806810372,	1622455591708804434},
	{1000186615386085832,	478107832313010967,	347632954881502753,	578536757312715338,	1032748576601015661,	293173936243394884,	1848894683875480820,	842785707969578566,	1676843667556853697,	1204977143858295342,	1621823568269511273,	311104289167338389,	357479316996815966,	347596581715575125,	1463316700502836312,	581027912574685130,	1199626470790462410,	729887369739348356,	648282105506669901,	844518099453838493,	154298023442799711,	1229777103758402729,	395860750784059453,	1369619448453137268,	1593037753990317598,	1292665274504387679,	1043181088976000123,	344718154412734853,	1256575872560234367,	2305691383501328584,	182045084765536499,	1196071100424529891},
	{2223388997484546514,	1453702388344291166,	1471338160370179826,	2020185193527010171,	1908313466221519015,	2095393772420815362,	1137928419027414073,	211036347048612650,	2219875955381817232,	2089420125254747127,	819080877057025544,	1787575989281264348,	922622361714973489,	1515876616458839718,	1273712113607821213,	369790389993520115,	1236610476316344593,	862047925307757084,	421126303440641573,	1924972647222094966,	2255347280865915476,	1830504026298277415,	1163346142838693922,	2296265897086079096,	2004186515453205599,	2167607609301611197,	574206933078189129,	166759899077269993,	855048706413268241,	1072539768751710332,	728598982630820551,	1091750030937956577},
	{1616345787175791699,	628916825724408436,	117085281079503627,	110716626997129840,	2238075906562790683,	1014318183384276362,	1550601020140224306,	1509431440895906723,	286057718398772169,	992828450669897604,	1528591667346708392,	83320228042217872,	150302270239895929,	1648242869511989453,	1808720332590847894,	933284013979591477,	782460501246376840,	1406368135080398580,	1539141788318570384,	2084075095760974956,	1668336680933642716,	331246166876933993,	917781813892224647,	1710175688159991144,	1489519841426653897,	2061970346347980821,	1708605415779868042,	374660887460901369,	644088997741424603,	429298069021362645,	2145231163215786031,	1275152880496104603},
	{1068561148895187220,	1915848428054628542,	516551270028526249,	1228971276952526792,	1389267142902532205,	2271443216124120950,	1975048085344179549,	1711010567126927785,	1458747513000821031,	109668491258441266,	1867097501101155426,	1532115216903568820,	826127976170746400,	967348476324627453,	250871764670742586,	1026977415990039935,	2016069289619899246,	229517932060441702,	130725337944474317,	616605340249602509,	881896416516399774,	1826458297946638553,	265747505678213199,	1699506703900835827,	420952913642222331,	2023663357264235421,	766630537914674696,	1882771001648692898,	922538104815838841,	426635289869493588,	2091899378504708909,	1256020334587057560},
	{1218946493389579583,	2287901713925263017,	1749183908022249342,	295587785659650036,	1981001321555445152,	1614776716202966560,	383252043949965442,	1849621857463353248,	1299748564378571704,	686907435399194561,	1048220572409213180,	564089603757747481,	1228418881873586453,	1781770567195947089,	2061587452290297307,	1530877812229635210,	2062889418494939191,	699265009775696038,	454694950771664375,	1856722172502843926,	379449439007921700,	1609763293277613338,	818227425242375893,	620627430682102624,	1627541663900003497,	1566245608158326484,	1474262221245168252,	963168074309846483,	2101680488570211616,	121231935166252110,	423972510717624531,	2038567593793631787},
	{1788517301512588659,	1273245457726076596,	1177159077568181246,	2000206164554158951,	1018595783092345467,	369144262604223703,	471396034213783258,	2290524861359448297,	1177907701735757357,	367582255671736003,	1900257904800843580,	230526420848045353,	231709197978819772,	867354405345713402,	103999023115000298,	2223491406590121773,	574679673270391518,	1420424147285246502,	731138747200513868,	293265406910077362,	2113709259434994242,	1130579514113903141,	78313557514668705,	1069401835634443238,	768096326462347889,	1994444495461326357,	1154758576764868051,	1200689294477318555,	219484684378975302,	1641464293468174008,	2121671589676704583,	1061888652144234104},
	{1709604545136707596,	2171011642999593216,	97277630778511459,	855099527174005798,	1606250347999801208,	2145678022738320221,	375183074517688077,	1037888386352306841,	890333770671350580,	1657374398860521351,	2295840044607534672,	1341015880188886095,	1600650784268870856,	1632957751621823747,	326072390348293718,	1276211293693248349,	637707876651941679,	853866959687860261,	1086234117230289059,	282297009460077280,	1815328668106144585,	2097275097048970391,	277444902957730663,	1946848305684277351,	957060125076029352,	1923597687736348669,	1872510298752568283,	2207993851149719146,	247754432990382734,	1637949291180674954,	1336060938764932530,	453744721889620205},
	{1144992648027306900,	1206519769924198758,	2072766832595661866,	22791003560240106,	624277913144247237,	2061503602694955907,	477806339171080803,	2073887881099347506,	258205681023064319,	1466756123493600508,	878085082858478862,	1362942146396384229,	659387891902504720,	1879666903441743438,	1483549800977348575,	1624491499186446177,	2183527729507062762,	513936355362719758,	194031372229783022,	703207872262656157,	1098846190893877688,	400548358656375537,	1168613519919744066,	1633020231399105864,	2053459895433325701,	1701819642645743472,	669023555945854353,	523277283207777584,	1355938172238648693,	1305342225415306363,	1822120710717664322,	274172286620698426}
 };
#endif
	int i,j;
	__uint128_t temp, Y[N+1];
	__uint64_t *ptr, t1,t2;
	Y[0]=0;
	for (i=0; i<N; i++){
		Y[i+1] = 0;
		for (j=0; j<N; j++){
			temp = ( M1M[j][i] * (__uint128_t)X[j+1]);
			temp = Y[i+1] + temp;
			ptr = &temp;
			t1 = *ptr;  // the only place in the entire program where endianness matters, here we assume lower bits of the 128 bit integer are stored first
			ptr++;
			t2 = *ptr;
			//			fprintf(stderr, "Upper and lower parts are %llu , %llu\n", t1, t2);
			//			fprintf(stderr, "Will sub 2^61-1 x n, n=%llu", 8*t2 + (t1>>61));
			temp = temp - (8*t2 + (t1>>BITS))*MERSBASE;
			Y[i+1] = (temp); 
		}
	}
	for (j=0; j<=N; j++){ X[j] = (myuint)Y[j];}
	return X;
}

myuint *skip_1G(myuint *X){
	/*
	 * Skips 2^30 = 1073741824 steps, by applying a precomputed matrix to X
	 */
#if (N==16)
	__uint128_t M1M[N][N] =
	{{810597818204201665, 536865238669308213, 817150683785840422, 		299613084680767909, 1265354587553896858, 795781700703133231, 		2163229501001659947, 1260507935229122961, 1087065451188767954, 		921851520152722997, 1292111400149670926, 603612135862697067, 		1459181315038675418, 1187556754581793368, 1609738974574022159, 	1244213075463471794},
		{1207377343044121050, 854013949460027318, 		1353090275069948780, 1659116337504639798, 587780365352856721, 		1609981353346649893, 1902062568650313065, 2248647173959627011, 		1508658268305302550, 2172812208302400139, 268043764436609809, 		671303693271564856, 940986965496800654, 1881659300065957173, 	1453742584469647138, 638542470814804371}, 
		{1170541610624770306, 		1331045337975123100, 137204881791788848, 2220796546106906965, 		1393485245130018046, 988866616266273979, 1172073314234562575, 		966617046619067359, 353687157940325174, 1126158440242693380, 		2085292948970235529, 1986079115610204808, 1966861059417878100, 		1448568105917260577, 864359235181522557, 	1982096461270240352}, 
		{1702540750434736241, 885878460676808471, 		2079602216921338061, 283583482320249737, 821828090320705276, 		155445830117780363, 2017807296342376036, 2044933318497852432, 		417985545825403342, 1483647517834408671, 1333619749640973330, 		997196892222202235, 553315487515542346, 1686465974112484702, 	1883218907285546094, 1415277566579176067},
		{1422985039599232130, 		1412770161374316682, 361812359303506277, 1623802539032044878, 		1950953988646773372, 2050560725296125550, 1145523596068528209, 		428743615579159742, 74810312647234541, 920694757193540402, 		768609776736885074, 1255809335960320743, 1525608701994604556, 		1425288944999418012, 31244755632464368, 	1607995891988687018},
		{1430692512619288193, 835736388458336949, 		654408547688131853, 2280122114111916056, 802657211146997429, 		435205743057396334, 496140613008507017, 1589004487018735677, 		383404293854834205, 744508732020432601, 1293753072844050099, 		1328026690473103497, 84600359972084021, 435300797976259616, 	1125747951072797874, 1442206959500710399},
		{1253389133249889368, 		433498589706896656, 2212104821440556423, 637703652563748338, 		1025364712747105330, 898149057066984753, 1147956773285230257, 		631679497982846535, 1054844952619877135, 1768140454325555862, 		820926943948253935, 2039528584665977248, 732034833974105581, 		1381036537137324814, 556407591111893923, 	36891212970456897},
		{1064571306999068337, 1753662122889621276, 		1877586369370033190, 802326598209569919, 450736754172204352, 		165262796738196753, 2228159271602665275, 706283513006075370, 		1733664553424483884, 2052633798746485040, 536240717710476873, 		2236059755664286899, 101769359846054267, 1615056035589891132, 	1920940714748547706, 95051513552874416},
		{2092251401027679777, 		1037437192249920713, 1835721683362468485, 2225628161672311550, 		1838171024657468451, 1223342829759625135, 1379542132604772423, 		14802536508674979, 1046083874954811004, 321414181022368394, 		896888488603413108, 2292563068543927839, 1557094512735795875, 		1499727978683854399, 1462495633837948314, 	1364079528489782774},
		{1783608279288791187, 40783284054579168, 		1228690427944762755, 143902698280484045, 1643432213052136303, 		41695740170717534, 573452416691661741, 350235344977604063, 		601916986286442592, 187625777415538341, 1728016942352033971, 		953592481985999070, 1484153241067188776, 420290974912499261, 	1138382618690544177, 1080540315472533428},
		{2203137030087799600, 		1872829312970969458, 168914863203669910, 2265485240513339735, 		671329786122117991, 938520601912503667, 538390198521541663, 		1470572263718982366, 247928022946613698, 603729350643823604, 		1553446388400448207, 434822167787540318, 1585911842863628933, 		18955992897334611, 1512505814505365919, 	1520210954034782453},
		{1019890735489371821, 1258982382836889970, 		2039871437586972671, 2114372909191733128, 2283544603628030647, 		1039922485214912724, 330978120540090610, 1835935434225358784, 		368837772629681011, 353860073692366087, 1027715199042568373, 		2212922246713938937, 686812654893599746, 329040994859984691, 	2074159895017042871, 1610910280313705260},
		{519570516943961189, 		1937744668746657144, 231378270106112075, 1152059386519707992, 		205033589433051734, 696827629485235250, 1762087832590985828, 		1227768064106443958, 640764225556632904, 1682436365210487144, 		1095899719744103236, 1922681067270805064, 2057427362612567063, 		226923480993945785, 672431112846968565, 	2140473254848068198},
		{49435374244837884, 770205300696099350, 		605954995317626784, 946564558082312247, 938138848380947695, 		143869373981400393, 338841536883104438, 69175333046933976, 		2164997083193333162, 1573361353897651198, 369947940745111453, 		679889698420473881, 1689923423754550400, 1406315134425105006, 	37150022059613723, 1327674270727753940},
		{1327674270727753940, 		1077039486975615779, 1490975227147360816, 441704689939154218, 		1287174120846997710, 929713138679583143, 1655442908795547042, 		533813699784808483, 2105352138424457465, 1954975824126038397, 		981725624601969234, 1472377536806265659, 989831456095836309, 		394114139874932968, 110505850545487574, 	1948878891398374706},
		{1948878891398374706, 706469650057133174, 		826404703223477618, 1904910967319105853, 1698277161944641571, 		2132643551754841202, 572252156512168576, 75329669697816990, 		1718027499987763875, 1371047567850412496, 1804599509827619329, 		8475425077900071, 1963029449010562084, 507285375385406959, 	2217411068378197569, 1909063484373774566}
	};
#elif (N==32)
	__uint128_t M1M[N][N]
= {
	{78631879337802267,	1375622574694180789,	1596794735315537715,	1293933980824896544,	692884451314635684,	2004109640436427360,	548719604741986250,	2186890926293172025,	1848300286524118676,	2064946215060388364,	1849919407969142279,	169783358607220653,	1110910827936006366,	1087622187925622110,	422970997447644681,	826827775982711407,	1366455510341692558,	1722111568372235843,	1091144068293688790,	2052373983366892583,	675009385261195836,	3525021939126551,	2241581498576885638,	308770418249339558,	1334424513904700696,	1687616631217596337,	2060942777960371487,	1377334365259367969,	2276739182456171523,	1554749280559411436,	448262067712426480,	617872989466225045},
	{423126735246141984,	196233070171007323,	1389652795739233975,	1927715931896285258,	250982894282958539,	1907443107536355900,	1276930970146921000,	2067876961039175657,	952110260229216286,	1467936035073631461,	1519244519250660019,	132809056497913264,	406121204763364812,	1606875160258678940,	438359033835250736,	1026430919945529158,	1494119886619499210,	39540664371030928,	2005092385723153587,	695875787922320485,	333359011516158769,	214306814095491444,	1455749421397346582,	161312350179789927,	2040107053440868851,	2138062657761169690,	615723762974065224,	600765454174312779,	2169260918917834901,	63764756762201611,	355393484192702441,	278651145958627915},
	{1113643437202329217,	1174214134751298432,	1580806678348115163,	1838926781834702518,	189499698292780850,	123344522831581679,	437607708411881964,	1370281851401173439,	735421367250491128,	1827822747929496918,	1293882252710985989,	1716838650335097420,	2245107670138017081,	2088537576673734336,	2169900562543051463,	2213672992494367156,	1667755905552183570,	2109477747196020231,	291328222281040649,	2225686684192255285,	1532332665231892186,	1419444728772342034,	541337810559868415,	657587014867880493,	645256685517953460,	1276659088925666624,	2113371594087990193,	1638234503871237343,	1800121250541021774,	854104607273384673,	878623242178685737,	262524900672978402},
	{263992549388583539,	17593894001254650,	536870658057686801,	1227389125997478012,	682817691037959673,	1921340759516493519,	1408822007516689926,	2172555067692878443,	135139670281526056,	1903435608013574842,	982295732541689763,	2082725013511711102,	791087031638416294,	587148828318244963,	1701889539551208478,	833579086013627224,	71559089990096209,	1353889133364375166,	866386342243882402,	2165043872894143172,	1140837668521970731,	832020765795445328,	1238578221856959004,	2158416809478419335,	78221400807060520,	1308424131271843932,	1060598935734955844,	1705942351379748130,	2106708286771275049,	1295121562354489489,	1644444457784567735,	1401853000164669033},
	{261907975880062330,	26569885437396617,	638798109299955373,	110779556135005927,	728624474502753415,	841397478845253324,	506659273907368152,	1905667570040718847,	1431298833459939806,	2193612392148990116,	240680792558031346,	808156126232103877,	945603040091814217,	1748437712708997601,	1853966023065485744,	10485947740827996,	1092936388135984450,	426544587525387657,	71883891409072515,	1851310272499108880,	2261176082791493605,	112108094773262200,	1780716526948169075,	1010993241864095577,	86167850574141561,	1124775115486712847,	432521105663564163,	1830696046922801037,	1347722807249986628,	1595922571566569767,	1736138517435594305,	104422664176755782},
	{766231920554145363,	1959239592365302832,	1352812773145370592,	364236660203508649,	2293590769078487304,	981325533457052864,	1188219808946558571,	309142410505410062,	1013788280712773746,	388796489036988241,	355977990649316877,	534009814227922315,	1568953775881471701,	1298477303596752799,	1496835558648293458,	83944520196836677,	467689543027604295,	1747986276587179822,	1381658866892486953,	958274682884879807,	333324915981803035,	1434916169073225433,	1794160850299128338,	1890283267633483653,	1457711703227588160,	746010722991710550,	1397907145642198498,	1854688191582815234,	1599090098359764035,	1173839450561644956,	1896723580778650045,	1827832577086620875},
	{808933651743461608,	1869377180174363542,	162355485906804737,	1044487690947788329,	22147944904790847,	1732445134751264274,	1912176215062001306,	397255832841456257,	1230574330910857475,	1442688401637919722,	2289099075310231758,	1382531546268089248,	2293787903795576088,	2021869296153717472,	1851434080636305759,	548508257574409570,	2263154430131411919,	2063062758396284529,	1977839809659650967,	886251653670329947,	1386912474178452454,	402742845872560223,	94168933359314030,	2042299641772827296,	2276994109895566991,	1784414926388262253,	739574296895489747,	1111902263294271241,	2166354350181403600,	1453737669471269458,	751756329556720145,	2057308644121705785},
	{1279200368099242397,	1298258666156145653,	1744881949379513453,	1710249130775602654,	2142757053669272475,	254932376432782990,	453316622920777639,	728911519009773545,	1127533426055797000,	1769697596934091548,	2092034812759241892,	1687489311279632581,	1458601471801812474,	686149665295368767,	659520308094751010,	879622336692513826,	1859055079974403543,	399195155783130672,	1351457012758429703,	324774785311470170,	381371827409819310,	2254358039577552484,	2068216046780003095,	154176552460723275,	1914176088501946828,	1634036161091722717,	171795805619899612,	2132458499882373170,	1877832542402271393,	1259701939752711823,	1733635888380893960,	1912632087548484196},
	{224232704049351096,	762120182287782440,	2080616289334235645,	1128249417316849575,	1234933430995170932,	93856291386924885,	536032438767483569,	2294645259420665432,	983147469634388990,	905727734090687147,	786962946980637769,	1783062110944706115,	1671153671681160204,	1349860267071144494,	2076305494334240128,	190494935804072292,	913383336054948323,	616030256299278564,	2082947791994855912,	818882012392347107,	912689808927485775,	2062081700902361157,	136646202736148246,	779005190617139649,	2127433239460975040,	1377126997345942924,	1202895927583559183,	922134459371935185,	2240251604399560893,	671494700285866323,	1065666210034154188,	409672437991373824},
	{267409637232164299,	1360696848797900847,	1246285648382647948,	2266589598345876669,	2116828567402795501,	1381564421733800555,	130176025932837770,	327571845524608760,	36395436435028021,	824097891465886625,	809434269718792270,	1577110668190301794,	1366487245964875991,	854339581319972333,	2130816300293130683,	2248061731739341145,	1003032782904953044,	198855021680581379,	429998451225669655,	27131545635912854,	1851510538467931489,	84188828248944983,	1030212875266431677,	190620388305693524,	2176164417919093651,	642651997850937328,	1250456614495565355,	1380695395390967579,	527394741234716404,	1657388134476329201,	83287460819020823,	397696531687414416},
	{247746081458254374,	2167868465288493317,	1719307350948315181,	1479341186588611108,	240884356532468014,	1688524952117420167,	378326200424420264,	393222857981858254,	1062126102865524537,	1457910764888803035,	1520335117328245067,	2298202185815404031,	1458655769960167340,	1546152314484437207,	2030483808720282636,	1396305712658071149,	607284319064121060,	1520626128881020024,	1948570028007082283,	1809617259738586421,	230499390593301542,	1225617702307124483,	1966747145819565714,	1215512317616214717,	507245087435714326,	1412254588877181253,	599613581377150244,	1291050164398368756,	1284083027389165985,	1798222884732815278,	337438559453098128,	1406751720817581409},
	{1082125671694226549,	972161770960172098,	1716241016235999747,	457441950692913707,	1028397857085538951,	1364702939339732355,	2058622723803630232,	2121759965106220685,	1311200546082744661,	2163527221726381574,	2239197674156742095,	80828700788070794,	236264445852137675,	2064921206691156984,	273175110213161328,	614228181991114526,	1604148056048488414,	357903409306481724,	504270963389674372,	471987699911584026,	2088519813462061043,	584936653082862098,	254542054233358650,	447880541315107268,	298322675471208736,	662084113229375513,	179999460292722074,	483984749895712921,	1019198951532690709,	1795368001971179514,	1939057634989301355,	591589658087175433},
	{862261128647469403,	1085833533290699946,	255161804651201168,	733868532670551457,	2060405393031112552,	73582052837620896,	637095757250325855,	233310587938607167,	1339387947121762036,	1134321831226634907,	708368673153305029,	853436676956343896,	1054656132141142620,	1940014039307853732,	254468938444211552,	231090787204240294,	1946315162702440568,	1754636664867029282,	908261536701202796,	315499374456768217,	1571026850141269973,	1701852550984264364,	1763007874851402135,	1275917136630510606,	1703654301400175675,	1421216828572657580,	447190384424918063,	982467131268266353,	2204548923328238802,	1868656084151993448,	1792513119209543750,	1234833701311810631},
	{1417197752706197365,	1844963017383445487,	741686439952484766,	283867584283741738,	1773652575501080887,	231167027140278616,	1981607143731258981,	569067102699620623,	1922565414702219768,	888583600206057985,	409591852251336559,	714067263775784878,	2250594855401701793,	569302048465092637,	1571399287713731660,	1327691703502912840,	113777631116869953,	379601869816373844,	776564479490048917,	707602047766214945,	102951985519698670,	46892922037444932,	1346094966468104912,	2179264493264237145,	1087055291059028670,	70156850172143396,	1643013791442532080,	1105678038969124513,	1496974769381134974,	1217024463218428926,	1941944166332807382,	1645968603429786145},
	{696303192976347039,	945042696642279331,	2223794025733327286,	1843495549990463544,	152937015513464773,	472007395129997527,	1762958041502639783,	831321072348649279,	642366623698800402,	379909629936966725,	559919386804949901,	476993404424418739,	843171777499469389,	1111645704403712918,	1082600839358858821,	660768611881692149,	739050595993826481,	1359651581399761527,	2132220054054529765,	648187014915953222,	651506607553011406,	203308494030361139,	1771739867162594911,	539860422807176420,	150538787529545151,	1830151743588514520,	884004197245453424,	743068513505583685,	953142837764715224,	2203379100030654385,	565392842284864404,	2091375213456071014},
	{1174854659506459030,	943915845237797098,	1912423057229097483,	1473880729343387840,	1547702856386299825,	1998125206289108251,	852342938624979941,	991838114130243444,	439367288986466303,	2091002545188948622,	208151239255128306,	545066818842381147,	379886716326199051,	519853887579345502,	1603959075561454019,	1553928790780574839,	208212181343469745,	422578842426468032,	1786677133195847006,	1233652973408544802,	1285273327411772942,	1981774672757854385,	457645106715690677,	555566147408104008,	1219590536172817057,	1232881710781824468,	945898128349893353,	607492628908595013,	1089990654768815258,	104473112103323192,	883890727629185893,	1494684527450615377},
	{2301429059329119517,	1403435217883959777,	1153464309731348533,	1364139625629382201,	1127227162533896722,	1150188416371481308,	1663751144697521690,	2109102281160518356,	1970633717360022662,	483069269594513764,	729770584914895324,	2269131727816248031,	70194345499801449,	1833091466952953294,	1443738269539283924,	653571459109895811,	1487668615669436356,	528248261659415183,	1379122756625717357,	966896400584566226,	492658588831968205,	1472041254976493979,	1025042462682453480,	595687083093253280,	2047823819917720117,	1218532758612714687,	148423232409340411,	1967077796057240357,	620520523446453481,	1660102782165122053,	311410133389685950,	1202388612973507382},
	{292580771255616390,	1688331081972910954,	2254501714007735855,	340205828271655303,	285906094309711023,	1623256675917739435,	233918873940291300,	165586121426728235,	161855901297086982,	1709031372830768598,	2222005609141652126,	957189205855837444,	662988481137012546,	1944466343911298079,	1822587703935498360,	498846361544435634,	2020018268785451390,	1980979400917992046,	1194833751840581125,	1755749822788799916,	234088256308318173,	2073879694069742647,	2141872528673627875,	1015594210938428226,	438748603227640169,	241689187676877786,	2219007290118520297,	927987764579741737,	199184183477002054,	512431518312465353,	909889443013226963,	2044772548363879958},
	{1685028033213043631,	1103714558672485048,	1617853624949162882,	1262480303836463633,	1475264498602498040,	1328413957077373724,	906399878855528173,	2022317831460752735,	628863041311056658,	729813111715481251,	800705128983117249,	972259791112905017,	536121958071095057,	1426997232246763257,	1598211847719302871,	1140582474568962878,	79727810806212960,	2121177366847144414,	1689986811792822404,	2252106092622769850,	1187824428976400856,	2136888622173462057,	1992416311348562022,	1309114489534132729,	1665369393617157139,	1895237448450711106,	349115491936474089,	1433385367131907293,	883022395649808354,	1714912547160638963,	1670603263673502604,	1508368752636767976},
	{1836900887640209110,	864935930730553116,	958239575504329311,	1397381115636599480,	1813556695382609051,	1442358611363946139,	220094319832607221,	1712460410641900380,	1846039937513035714,	2118873342744988144,	836033494951256003,	1921472139302318102,	2157696620566359881,	1022219855549053221,	1635184276092928874,	54997422389935362,	1042001862490457293,	301685137387424464,	158089912734180447,	360647501905758063,	841998141280146394,	1241376929317246955,	341193256990010371,	788331972916533226,	2039546861023264234,	170615944702997285,	2060927723826708363,	573069738428872300,	361072773889622768,	603129095633832310,	611550566795118622,	125474075120084294},
	{2278212548476300882,	1229790141659206319,	470806541251038049,	315455779713454914,	332286033367414088,	1438133148511786800,	724417404860853538,	2198374553284882473,	251274327332690775,	1405788674056027476,	2054203523610734133,	600473473446973229,	329585085974551701,	94328693671066166,	1021072122707895529,	1424701766564447873,	1177246828662340869,	1079238277900079941,	154200641625458256,	1826208375323211998,	414537718543497604,	54043516684554981,	1917758188998911981,	1183808132323098158,	1371803052801043645,	170213640089459616,	232545960963241854,	2019097480971200244,	265455006225457851,	879308128534787378,	1797188653320719608,	1858340879130428591},
 {1002440129222419255,	2022918464509034269,	1326312719705675884,	1057981086823492246,	2132588666649237705,	419960871080777160,	1988906370809335195,	238062046745348978,	1553226612815292052,	422164529702719538,	2187820342253093240,	1377093669925425721,	1787656796794333130,	1989995776783343953,	866831983756556483,	261383426506339904,	115806883689817624,	2048893675068062766,	479452546088207136,	1407956682977372805,	359758986224315594,	2039434742135353157,	1785949179250163549,	93502236021076189,	1304367369461726689,	1029450447193790288,	1354841649808757520,	234028954927899142,	2063893711756022939,	184158769195401887,	1155487161435742446,	676983730632626643},
 {1827282911009418813,	25813319660900167,	324958929353262706,	139714080561357144,	403452272176094520,	1312241159632353413,	997434028547339455,	287236454231217439,	1932763590132005928,	976091621630808471,	1402847354177417564,	1061311313791729577,	2107028475901700758,	2060343085399523303,	874957663184932047,	170095475687936305,	1933442115784349124,	157133126693078411,	1259899314668862443,	1807092566315373246,	503637672866096525,	17570355100659438,	2036003465509879243,	47180963433607153,	746064766579493140,	183090184226238109,	1435091691543760436,	981496128789399912,	1791697928138220589,	1547525668659204124,	1794852419069710347,	513785669550765284},
 {2031467676936992743,	1680880068468088818,	709634444139759265,	1084717704253058514,	539987260291573549,	1858527446283925056,	431052996437510875,	963427553904786035,	1920893421524228355,	1199532972117676859,	86765376786633914,	949355049228064375,	1280835732354386862,	1816142507050740291,	1205884859092412196,	281079324169121739,	1984918314938629818,	1585186095794032979,	1725137852922879285,	573217109100276612,	1411931144164729425,	617633838676085163,	587751317844292870,	873953658408926070,	1767692574261951154,	1228682448464070240,	2269602733230880889,	590789526775687946,	1972080329486306491,	1738522464902384187,	605049558909312410,	128374667489984297},
 {2129643927517098022,	80837252874200761,	231865072097528800,	1739469345347168033,	78785722554386291,	433693126377706649,	190864004348360720,	282757284974918692,	700842605622155089,	2067909566879649498,	2226795880842848214,	724774429867457230,	373022451125370245,	2214489126800035268,	1668492255579303619,	1765042582456649959,	1993273109913407573,	1288640037962814022,	540885561832717532,	374375067984141468,	864216754005080760,	83248846911287591,	152116087572906808,	1392377862107996679,	1339413544823592910,	978100736262940833,	1472724099795166449,	671173486888430527,	544076208072553663,	679639938200175520,	1929519261145564250,	1721089707962608424},
 {1770934836254798949,	762973097033260349,	1603969025183011348,	115457594658322090,	476819489261345845,	865261001295108089,	2112937817773017145,	605330810647158146,	1259986609035105727,	612043192811466154,	1743294595902850485,	550303030164078833,	1345465693856483054,	2190737577206116023,	304579966088490516,	302167194841344738,	779077746830188385,	614992662129892542,	1534200428585293707,	2076306877038992349,	184294464908041017,	193055902948075710,	69609442050719325,	590355028954948631,	1464229200608376516,	1017656335409685744,	1124178155434136822,	1520837286806462218,	59986098039778476,	695803291748539130,	1926600420711660804,	948145954168122139},
 {2093898205505417675,	1918696005131781821,	119323256057369815,	1099380580136113644,	1528531191217427343,	2259773529251684588,	1338850793321369826,	1380872710430230123,	1032576798876480627,	1137249261433563654,	455367910419070943,	1754559002094859076,	2249324693226865396,	1816974746779336405,	1316040556971076947,	465095086329957116,	1989522496062236819,	528067166471046219,	1491251949195123241,	735315602879581469,	1248958919634987007,	1025522162389824735,	240829139754954245,	178795320525044661,	370487844521557480,	952643493241214377,	316242130089268632,	1881115334835800560,	680755449384799974,	202724747025960657,	711966645296902740,	1923681580277757358},
 {201927406802132780,	1914396218842128105,	1411930862928219809,	2055285821839830276,	864208138320544368,	1950955720240877489,	1321668547308246137,	1294894046621925138,	2025024324374127769,	1893634038534953438,	676258079138357398,	1995903837434747593,	212680431506815883,	2078231907415380349,	1003280122233512960,	563348328114469133,	1105038517588968839,	1582054057359368184,	2269983878638145066,	1792983077250396285,	2137662684695839795,	269315022244121099,	530849679340003152,	1509553946062126495,	301216045451369874,	835196478807599057,	87803459966765719,	1593819278074482379,	1373489442190870365,	293272169095364320,	2015489211517076135,	1803175879095838627},
 {1727492590576575784,	44822412662655888,	1779688217226463891,	1649280001900453393,	594021534001836983,	482695013673821009,	765504941923980625,	2194714981831584811,	779100653154353223,	349285923495086293,	2292786015545785577,	1935553215455227932,	668115173839118060,	1404824684117293961,	21927180020433650,	10208301203136958,	1635739924124018894,	304035269135362687,	387515999155024515,	444834059276052832,	146795319684632724,	2054626106175442249,	1082885762594191915,	346820633876272416,	2178614058041186724,	1516915544483165119,	496043833371454019,	274338202368924301,	689311614893585796,	701449185832794990,	383819591164767983,	1013168768523555579},
 {1578597107939950517,	1355435449988677986,	1763656134840031135,	740377644769490697,	1639152727227967372,	2035031272494858234,	1440384606092892919,	1560491895865327358,	605799445977245652,	422905984126940771,	1312634986319302865,	1514489154969692631,	1425507367199865457,	975812140924022643,	828074365676625049,	924731270534042525,	503045294047036759,	1549022313802268735,	931603296945468521,	531347103765885901,	251803078646173887,	1406921115990638395,	46228258880463235,	864205346081550973,	93488846314721268,	1002890370939024199,	255611680460738775,	618431759976440320,	332579313176738321,	1658668277146038541,	1109626202570225660,	1057992980026153782},
 {545699225200831254,	277713355746960520,	2062950438206915421,	28353214177237171,	231135948429893076,	127900072029672338,	1192677030363778640,	1818583494799755526,	447102979677189694,	739736137493086308,	2119596500228701510,	748949112942941492,	874578312372845891,	295018091755518812,	1014937083192563941,	1727469248759191627,	1302374064841143564,	2233156886059576778,	1126724062323372791,	1982444599068482344,	304067762247228816,	1424937438702915184,	2283096154557169601,	1373960109192379657,	2149655876250107316,	2291996709390823787,	307263550663415505,	1235760673925934700,	1807669293986442325,	1727499192036450705,	2066845293883469211,	1783799591431611459},
 {1154450414072823863,	1899655489806178886,	1894983510335157404,	1687502271406537959,	1153656666917174519,	1216391292404494426,	219638556848531423,	1631654357987848988,	2279973022151634449,	1224212448117369954,	1534498898492128891,	1134199467946390622,	1752273378403599539,	19114218912577955,	287200041623730256,	1010799452311149273,	47236059237088945,	129914153220484997,	1123895572258895379,	1346493748583265121,	71311554515061415,	1868549569690737767,	1588959331807672371,	981232396591805055,	1314290484474821187,	438708181447681054,	477929548062564415,	862496996893036224,	16171640685105311,	448262067712426480,	617872989466225045,	1008852313857315429}} 
;
#endif
	int i,j;
	__uint128_t temp, Y[N+1];
	__uint64_t *ptr, t1,t2;
	Y[0]=0;
	for (i=0; i<N; i++){
		Y[i+1] = 0;
		for (j=0; j<N; j++){
			temp = ( M1M[j][i] * (__uint128_t)X[j+1]);
			temp = Y[i+1] + temp;
			ptr = &temp;
			t1 = *ptr;   // the only place in the entire program where endianness matters, here we assume lower bits of the 128 bit integer are stored first
			ptr++;
			t2 = *ptr;
			//			fprintf(stderr, "Upper and lower parts are %llu , %llu\n", t1, t2);
			//			fprintf(stderr, "Will sub 2^61-1 x n, n=%llu", 8*t2 + (t1>>61));
			temp = temp - (8*t2 + (t1>>BITS))*MERSBASE;  // 8 is 2^(64-BITS), which is the right multiplier for the upper 64 bits
			Y[i+1] = (temp); 
		}
	}
	for (j=0; j<=N; j++){ X[j] = (myuint)Y[j];}
	return X;
}
#endif
#endif

