(* Content-type: application/mathematica *)

(*** Wolfram Notebook File ***)
(* http://www.wolfram.com/nb *)

(* CreatedBy='%s' *)
(*CacheID: 234*)
(* Internal cache information:
NotebookFileLineBreakTest
NotebookFileLineBreakTest
NotebookDataPosition[       131,          6]
NotebookDataLength[     38100,        824]
NotebookOptionsPosition[     35963,        746]
NotebookOutlinePosition[     36917,        780]
CellTagsIndexPosition[     36874,        777]
WindowFrame->Normal*)

(* Beginning of Notebook Content *)
Notebook[{

Cell[CellGroupData[{
Cell["\<\
********************************
Orbits\
\>", "Section",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{"Clear", "[", "\"\<Global`*\>\"", "]"}]], "Input",
 CellChangeTimes->{{3.746814597353241*^9, 3.746814597354561*^9}}],

Cell[CellGroupData[{

Cell[BoxData[
 RowBox[{
  RowBox[{"power", "=", "1.0"}], "\[IndentingNewLine]"}]], "Input",
 CellChangeTimes->{{3.747473487345798*^9, 3.747473493628665*^9}}],

Cell[BoxData["1.`"], "Output",
 CellChangeTimes->{3.747473500308508*^9}]
}, Open  ]],

Cell["values={m->1, el->1, k->2,energy->-1};", "Input",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[CellGroupData[{

Cell["Veff=((1/2) el^2/(m r^2) + (-k/r^power))", "Input",
 CellChangeTimes->{{3.746814722758141*^9, 3.746814723037732*^9}, {
  3.7474734958524027`*^9, 3.747473497088319*^9}},
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{
  FractionBox[
   SuperscriptBox["el", "2"], 
   RowBox[{"2", " ", "m", " ", 
    SuperscriptBox["r", "2"]}]], "-", 
  FractionBox["k", 
   SuperscriptBox["r", "1.`"]]}]], "Output",
 CellChangeTimes->{{3.556297172285837*^9, 3.556297196061227*^9}, 
   3.556972013131445*^9, 3.746814608071307*^9, 3.7468147258585176`*^9, 
   3.747473500446884*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["Plot[ Veff /.values, {r,0.1,3},AxesOrigin->{0,0}]", "Input",
 CellChangeTimes->{{3.556972017452288*^9, 3.556972040306434*^9}},
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 GraphicsBox[{{{}, {}, 
    {RGBColor[0.368417, 0.506779, 0.709798], AbsoluteThickness[1.6], Opacity[
     1.], LineBox[CompressedData["
1:eJwVlXk41N8Xx20ZiiwtIkIpkiKVlDgfS5ao8E0blVQqibRooYj6IkqhbyVS
lKSpVAqhgyhLZDeYMWPGln3fZvn85vfHfe7zes699znnvM85V9PD1/mEmIiI
yIhw/X93U3s56HC1Enj0Lpddzd2wqVN+k1l9JbiXbTr3g8eBQFPJ8u8jlcDS
eWbY94sNRQ+5h0GuChzvrY5PfdQOjtZd4cSOKghe0n5FMaINvF/n0i0KqoDI
3EuvS2mAl6c9b9lQ/0DwBseN+w59gb5CV+XSsj8g8syiN/fKWzBUcXpv2y1k
HzkLKuMxuHeq9WspVENw4nHz0yIvcPCh1OJVDtVQEM79anagCOdOMU7rFFWD
RrezqJd6I65L2Z/ZU10Nw2umo0VeNqHT7jpBGrMaMn76VuisaMbHr0tjV/OF
dtrShFAFOq7a/zlf17gG9tf07HKisdA8J1xh7fsaoN37EPVOoguv2T2xTcmr
gR730e/LErsws/lNkHJFDWT/zZ57a0M3as+WD8zpqQF5TzNtu4M9KGsiW96m
WQsiBlZDEfd7sTnvQcj9h7VQ/T5DsevVICruTM6RfFkLtvNMMjgaQ+jA+DQc
+KkWaLX6GUT8ECK/7vDpP7XQk+PJVbIaxldmi00spOtAI3WLdoHOCPoVxI+N
BdZBIFpaWfwZxXTHt7pekXUgofs7lhQZww5W7lHWkzr47WJav8hwDPeJMv78
/loHPYSpyt6YMTS1UKe+Gq6D8CWXfD3sx1G6OOX4vuP1IOWiTr34YgIt9mQ+
rTxfD1dOMiXyyiYwsKO41vJmPRTgsiOBIxM4JNFFGDyrh+GOn4t9YRIbtuuo
SdPqIS74tJJDwyS++EVt+GbfAPsd6Xz73ik86Cvb5OXaAOd0y1+lyk6jopIP
TeVMA8TtmzAKNpjGEE/91mt3GqDYuEk16+I0esz5xNpa1gAX5S+9OTY1jVqW
WX3frBvh3NfodvXuGaT3Kg147W2E4fzPtuvEZzEu5sqgimcj8OzvltYvm8U5
7C0j1243wvMDo9FDe2axKzhvcmtxI5hRI8Nv5s1i2vdC0VzzJrB+Ymy7/hoX
3T2Xi59xagL/ZTMPXe9zccn8UImlR5tg0Olj7vpULkYcsqIEBDfBY4MP7M5q
LnrxfsqYYBNEJbjOjVzOw+Uvtef3VTVB+tlLz9cb8bDFPlzuaVsTbDuR7hhl
x0P7p3aKXL7w/bnFzUd9eai39bdS7jYaSNa1FL/O4mFHu57yGQcavO86GyBd
xsOEiLsqS91oYL3bLG1VCw9lm3epBQTQwOvr+/r/uDwcvFyz3OQbDQyKKtpO
mPAx40vj2tzNzSAWcPjkuQ98LNce1jti3Qx7DTMWSObxseOJtJ64SzPMdla2
epXyccmNbbr255uh8dhcfgqLj0HWyataqc3AGleq4swX4OPs3JVBuc0wWhKQ
IaoiwE+6DVorypuh5WEnd1RLgF3zpVac6W4Gi16FeqetAtzVdFadq9kCtqdJ
1/nHBHjKLmxZkkEL+Dj5H2/wFmBI7nM1S2gBPdnQnAv+AvyaVLc00q0FPCod
lxlHCHDZqS1Llj5qgT3nd7j0UwU4MC2hYCLTCotHN8l7DAhQ8swyeaZKK2So
X5mpGRegBmOzXOjqVoj5/bFTnSfAPYVeshXWrbAq5I7tvrkk5kVUS7vebAWv
BN0Qs5UkNnL/SpHRrTBOozWkrSFx+Ky4VMqzVlCd0lMcX0+ilrORZF9uK3za
NZW8zozEKJUEsYDJVvDo49wwdSExNeqLqPocOty6M/dq9kESCwRVIj8W0IHQ
6OcvdidxnC1Kzl1PBynlo11BXiS6vj3JjT9DB0flgc/SN0i8qHZz1uwaHe5k
Bka9CSHxXnT8DDucDhJSOi76YSQWXaic0k2lg1dhE20gmkTdbRvGv7Ho4HPP
5rTZc6G9JCH/8TAdShfbFnq+FL6/WzLMX4QB7CUWCy+kCf31aFY21GCAhOUC
rnEGiSv7LThyBgxwlLPfOPaZxO+XqNQBYECqwOpzTBaJQ+FB8OYIA4K3aheF
fCcxXPGvVJgvA1YxfYfqCknUSHCuPR7EAEMri8R5JSQ6f1h5QiOJAXGJJen6
FSRmNlTcedTGAGuf8LfOjSTuPLJpz6VBBkhpLhJj0Ujs6nmm9o+AAVZrNJfv
bSVRieeXMX9ZG1hdf1rLZ5KYcbv1av/aNnBndvfqs0m0ldtuWW7aBv1Rl+j2
HSReW76k8fahNtAtXO1s0UOiIvVm0rGzbSDDKdqn1kvi2019p8yvt8GtwbAt
nD4S6bbfubyENvjZOC/KYIjES7XaJS3UNnj5j86V7GES57s9uJed1wYe8us3
642SCL7HNS/Shf/SmWJV2jiJtOnKXqf+NqAp/c2WnyTxXMjmTH1eG2zsNVxg
NEWitMyL67IyTKCrfjluO01i8sO5Nn1LmXDjaGOnzQyJW9UvypetYcKnVf9U
bZwlsS6N0ZxqwoSWenCR45LobWiTcsueCeVi8z42CVkiL8Pbw5UJ9S+ClO7x
SEzcrmJEnGFCgeavfEM+iUZ/Qkm1ACYUR52hlQi5av9AKfcOE7yeHnhqIyDx
JHtvTHM8E2oL2FuyhSziXeCalc6EZfrYpUSS+Hhi9cqH35jgXy9X7ynk9UGx
g+fLmRDl6L7hlZDLpfhZji1C/9hpm+qE7BHjeXNdLxO+OTnOGxEyd2n1DplZ
JuwoX9sh+P/5WtW5ExIsiMu5M8QT8n+PagY5Mix480zLre//993+ratdyAKD
xnX7y4Wsr2mSXagq5CN7FOKFzOscSsjQYsHpJzmZB4Vcmv7yZpIeC/bw+T4y
Qo7zPeB5byMLqJWm3h+E8bhvnG9/fRsLBosW0CyFrDdTpO9txYLUjHhGmTAf
M/mXF7o6sOCzlu9rcyH/DNGbsdvDgp8iH4+8FeYzxqadYezGgg7ZOQYUIevW
7Hi92JsFiR+X5D8U6iMWviOEdYEFkfK6qb+F+jWb7TiUHsCC/z7u+jEp1Dec
arfALJIFCTWO61YK9T9yzG6QEssCvoxnyWphfRip2JXVxLPg7hORFcsnSOwM
sw06kc6CdcqlOCisrzwz24P6n1jwZRXX6seIML4Jm00zOSygrC1SjRLWo8Ux
m76oMhakpLfbcwdIfG5mvS+zhwUfsi7v+EdY75cnthveGGbB2X9lpKu6SNxF
3S5rO80C1feHNMw6SeQrb//RSmmHbbLRDYJ2Eg9OWOqLa7eDcUL5uznCfltE
Nac4n2iHEfJYsoWwX/s9zNlLz7ZD0ILRGo0yEouVzfM7L7bD73NOqyZ/kng+
jLhw9VY7rBRMvfq3iMRqD2A+T2mHKxmBv81zSLyjbJo11N4Ot4IXGRq9Eupd
vS0m5287ZLbOusUkk7glbNvZ0JF2GFRI1u9OIrFn3GSFkigbPAPWawXEk7i9
emu0mQYbyu7ZvNolnG8iYcYn7x5mA79FXOvrZRKlUhaa15xgg25HZ2DrRRLl
cVhl0Vk2mL94FMX1I1FzKq0qIYAN7ItlMnrewvydVDaiPmbD+uqGdY7C+Rtq
MytRUcuG50SE/0cbEiWl8pKlbDjg0BYqvWkBiTIR0Baq3wHj9elVJUkCrM+w
TDy+uQNiC1d4vnkqwKc0G7ft0AEpi2NU7zwSoK6OY4vk7g5w7e6Vs4wWoO0v
98YInw4wrxleG3dDgLfnhFRFv+uAn+xtjFw3AQpuFn9/uqYTdub1fXNXEuDQ
Nbukz9pdIH7s+rusMD5K5XgrHRN0w0kzg9+553nIvuqwvLriLzj/4zKe6MHF
OF39y/4FfTDtJ3lUwnIWA7XXhCbEDgB109vzdPMZNFmo28y9OwT3dycnntwy
jYzCUfcoHIZTckf7+qymMFudkXHw9Qi8P/dBzsVnEoMmSl+VJI+C+G4D90+x
E3jUz9z/3N0xuN539b5B/TiuO5HblxU5DtQ1h84HK4yjW3ys7uHbE6D4y77Q
5/AYvro0bWXlMwkJak1K7oWj2Fx6WOKX1xRcCDVft1ZzFHWKnJIi9k5Dn49f
0aG4EaTxksXfms7A+5pMqT6FEbz4YULmtPEsfPGmxGyPGsY1Ug0jnsu5oC/X
yT2+eBgraV8CRDV4EBeRriFvMoTVveNVs0v4UFLh//7NwUEcTC7gbxAVgL5/
y4N5cQM4VXu3RpovgP2GVw5c+dWPS990B6v3kHBD8d/wMYV+3M36w7M7KUK0
2RaIWJzqQ0oaNdK5Q4Sovs1J+p7di5VFu6W4+0UJs55vG6JUezHiyYUmvyZR
YsVll6u91/6i849DceE7xYjWwxN+Bb09WCE79WVXoRjRnUHpPHisB7cuVNDl
mIkTA9XsLbmN3Zh1diRSKVOcKIgNyeXv70bdUGdCZYMEMWxgs/lQYxfuzPE4
8PuDBPG6Xb7vqmsXXrfY4r5Maw6RnOX1Y/9AJ6aE3c8xS55DBOzNX33/Wic2
qb8XIZUkiZHaX7M3F3RiLu0a1SdSkgg74Tep9qkDH+4sTadKUQhFqrHTbocO
DFxE/+/OdQqx5cXw/fxhDoox3b3jgihEV5XGyJohDoandZo/u0khBtzro+MH
OBhrMtT/8TaFWKsaFX6ll4PpR8Usm+9SiNiytyc2d3Cw6Z3OkPYzCpFO6hQV
NHHQ0PqSTcl3CvE4l/NcFDmYLTer+qeAQkivGPhyOZ+Dps03RmlFFMLozBfK
YC4Hbb3DE/t/UghZHfujrdkcPBLzdGzBHwohJiHVl/2Jg1FthUkeTAox5vzN
ISiVg4pp1pe82ymEacRW7ZmXHHzsV7HDn0Mh2l4nml5I4WCKRMNERDeF2KHf
qHbqOQe/6fbYfxyiEAzzuPQ98Rwkxr01c0cohNfVDx7VjzlYkj8yWTxGIeq1
zPc6POJgtSP3BW2KQtz7+2DIKo6De1WCL7NnhPlbbPSwMIaDdI7Ezn4uhWik
q/ibPeCgx7uI5ZN8CuFXYxiTG83Bbv/50yRJIa5HnvtrfI+D/wPxU0kT
      "]]}}, {}, {}},
  
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->{True, True},
  AxesLabel->{None, None},
  AxesOrigin->{0., 0.},
  DisplayFunction->Identity,
  Frame->{{False, False}, {False, False}},
  FrameLabel->{{None, None}, {None, None}},
  FrameTicks->{{Automatic, Automatic}, {Automatic, Automatic}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  ImagePadding->All,
  Method->{
   "DefaultBoundaryStyle" -> Automatic, "DefaultMeshStyle" -> 
    AbsolutePointSize[6], "ScalingFunctions" -> None, 
    "CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Part[{{Identity, Identity}, {Identity, Identity}}, 1, 2][#]& )[
         Part[#, 1]], 
        (Part[{{Identity, Identity}, {Identity, Identity}}, 2, 2][#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Part[{{Identity, Identity}, {Identity, Identity}}, 1, 2][#]& )[
         Part[#, 1]], 
        (Part[{{Identity, Identity}, {Identity, Identity}}, 2, 2][#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{0., 3}, {-1.9999989426838902`, 0.7452766111154285}},
  PlotRangeClipping->True,
  PlotRangePadding->{{
     Scaled[0.02], 
     Scaled[0.02]}, {
     Scaled[0.05], 
     Scaled[0.05]}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{{3.556297172785169*^9, 3.556297196113483*^9}, {
   3.556972013620694*^9, 3.556972040578549*^9}, 3.7468146081117077`*^9, 
   3.746814725905985*^9, 3.747473501368235*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["T=energy-Veff", "Input",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{"energy", "-", 
  FractionBox[
   SuperscriptBox["el", "2"], 
   RowBox[{"2", " ", "m", " ", 
    SuperscriptBox["r", "2"]}]], "+", 
  FractionBox["k", 
   SuperscriptBox["r", "1.`"]]}]], "Output",
 CellChangeTimes->{{3.55629717281326*^9, 3.556297196184051*^9}, 
   3.556972013674022*^9, 3.556972045704535*^9, 3.746814608125823*^9, 
   3.74681472593004*^9, 3.747473501534226*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["eq1= (1/2) m r'[t]^2 - T ==0", "Input",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{
  RowBox[{
   RowBox[{"-", "energy"}], "+", 
   FractionBox[
    SuperscriptBox["el", "2"], 
    RowBox[{"2", " ", "m", " ", 
     SuperscriptBox["r", "2"]}]], "-", 
   FractionBox["k", 
    SuperscriptBox["r", "1.`"]], "+", 
   RowBox[{
    FractionBox["1", "2"], " ", "m", " ", 
    SuperscriptBox[
     RowBox[{
      SuperscriptBox["r", "\[Prime]",
       MultilineFunction->None], "[", "t", "]"}], "2"]}]}], "\[Equal]", 
  "0"}]], "Output",
 CellChangeTimes->{{3.556297172824063*^9, 3.556297196221715*^9}, 
   3.556972013714879*^9, 3.556972048680774*^9, 3.746814608148108*^9, 
   3.746814725954357*^9, 3.747473501626356*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["dr2=r'[t]^2 /.Solve[eq1,r'[t]][[1]] //Apart", "Input",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{
  RowBox[{"-", 
   FractionBox[
    RowBox[{"1.`", " ", 
     SuperscriptBox["el", "2"]}], 
    RowBox[{
     SuperscriptBox["m", "2"], " ", 
     SuperscriptBox["r", "2.`"]}]]}], "+", 
  FractionBox[
   RowBox[{"2.`", " ", 
    RowBox[{"(", 
     RowBox[{
      RowBox[{"1.`", " ", "k"}], "+", 
      RowBox[{"1.`", " ", "energy", " ", 
       SuperscriptBox["r", "1.`"]}]}], ")"}]}], 
   RowBox[{"m", " ", 
    SuperscriptBox["r", "1.`"]}]]}]], "Output",
 CellChangeTimes->{{3.556297172846009*^9, 3.556297196262059*^9}, 
   3.556972013875587*^9, {3.556972050016386*^9, 3.556972073168967*^9}, 
   3.7468146081745167`*^9, 3.746814725984726*^9, 3.747473501913661*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell["Plot[dr2 /.values,{r,0.1,4}];", "Input",
 CellChangeTimes->{{3.556972054871002*^9, 3.556972057249209*^9}},
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[CellGroupData[{

Cell["root1=FindRoot[0==dr2 /.values //Evaluate,{r,0.02}]", "Input",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"r", "\[Rule]", "0.2928932188134525`"}], "}"}]], "Output",
 CellChangeTimes->{{3.556297174479553*^9, 3.556297196621921*^9}, 
   3.556972015910306*^9, 3.556972074863778*^9, 3.746814608237934*^9, 
   3.746814726028675*^9, 3.7474735030257673`*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["root2=FindRoot[0==dr2 /.values //Evaluate,{r,1}]", "Input",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"r", "\[Rule]", "1.7071067811865475`"}], "}"}]], "Output",
 CellChangeTimes->{{3.556297174523539*^9, 3.556297196667878*^9}, 
   3.5569720159862*^9, 3.556972075873295*^9, 3.746814608262127*^9, 
   3.746814726054524*^9, 3.7474735031702623`*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["{rmax,rmin}=  {root1,root2} //Map[Last,#,{2}]& //Flatten", "Input",
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 RowBox[{"{", 
  RowBox[{"0.2928932188134525`", ",", "1.7071067811865475`"}], 
  "}"}]], "Output",
 CellChangeTimes->{{3.556297174530752*^9, 3.556297196768156*^9}, 
   3.556972016036149*^9, 3.556972076872478*^9, 3.7468146082869577`*^9, 
   3.7468147260795918`*^9, 3.747473503244011*^9},
 ImageRegion->{{0., 1.}, {0., 1.}}]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
r0=1;
theta0=0;
t0=0;
timeStep=0.01;
rSign=+1;
xarray={};
print=False;

Do[
       dr0= (rSign Sqrt[ dr2] timeStep) //.values //.r->r0;
       r1= r0 + dr0;
       dtheta0=el/( m r0^2) //.values;
       theta1= theta0 + dtheta0 timeStep;
       
       test=  dr2 //.values //.r->r1;
       (* If[test<=0 ,r1= r0 - 1 dr0; rSign=-rSign;]; *)
       If[test<=0 ,r1= r0 - 1 dr0; rSign=-rSign;];

       t0=t0+timeStep;
       r0=r1;
       theta0=theta1;
       
       xarray=Append[xarray,{r1 Cos[theta1], r1 Sin[theta1] }];
       
       If[print,Print[t0,\"   \",r0,\"   \",dr0,\"   \",theta0]; ]
   ,{i,1,1000}]
   
   ListPlot[xarray,PlotJoined->True]
       
       
       
       \
\>", "Input",
 CellChangeTimes->{{3.556297156172988*^9, 3.556297208489231*^9}, {
  3.746814620997616*^9, 3.7468146289392557`*^9}, {3.746814665645007*^9, 
  3.746814684988491*^9}, {3.746814732685041*^9, 3.74681480577577*^9}, {
  3.746814860452112*^9, 3.746814914860175*^9}, {3.74681494719519*^9, 
  3.7468150296157427`*^9}, {3.747473515276641*^9, 3.747473569451483*^9}},
 ImageRegion->{{0., 1.}, {0., 1.}}],

Cell[BoxData[
 GraphicsBox[{{}, {{}, {}, 
    {RGBColor[0.368417, 0.506779, 0.709798], PointSize[0.004583333333333334], 
     AbsoluteThickness[1.6], LineBox[CompressedData["
1:eJwVWHc8198brVRUWkJDpFSiNEiSdFp2GaU0jPhWCimVooFktKwiFQ0ZhYaZ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      "]]}}, {}, {}, {{}, {}}},
  AspectRatio->NCache[GoldenRatio^(-1), 0.6180339887498948],
  Axes->{True, True},
  AxesLabel->{None, None},
  AxesOrigin->{0, 0},
  DisplayFunction->Identity,
  Frame->{{False, False}, {False, False}},
  FrameLabel->{{None, None}, {None, None}},
  FrameTicks->{{Automatic, Automatic}, {Automatic, Automatic}},
  GridLines->{None, None},
  GridLinesStyle->Directive[
    GrayLevel[0.5, 0.4]],
  ImagePadding->All,
  Method->{"CoordinatesToolOptions" -> {"DisplayFunction" -> ({
        (Part[{{Identity, Identity}, {Identity, Identity}}, 1, 2][#]& )[
         Part[#, 1]], 
        (Part[{{Identity, Identity}, {Identity, Identity}}, 2, 2][#]& )[
         Part[#, 2]]}& ), "CopiedValueFunction" -> ({
        (Part[{{Identity, Identity}, {Identity, Identity}}, 1, 2][#]& )[
         Part[#, 1]], 
        (Part[{{Identity, Identity}, {Identity, Identity}}, 2, 2][#]& )[
         Part[#, 2]]}& )}},
  PlotRange->{{-0.35914158417417247`, 
   1.6594713078084322`}, {-0.5518638658822254, 1.3570452632168928`}},
  PlotRangeClipping->True,
  PlotRangePadding->{{
     Scaled[0.02], 
     Scaled[0.02]}, {
     Scaled[0.05], 
     Scaled[0.05]}},
  Ticks->{Automatic, Automatic}]], "Output",
 CellChangeTimes->{
  3.746814629845565*^9, {3.7468146663611193`*^9, 3.746814685394573*^9}, {
   3.74681472617162*^9, 3.746814806903515*^9}, {3.7468148615942383`*^9, 
   3.7468150304947567`*^9}, {3.7474735042995996`*^9, 3.747473569830946*^9}}]
}, Open  ]]
}, Open  ]]
},
WindowToolbars->{},
CellGrouping->Manual,
WindowSize->{1918, 1003},
WindowMargins->{{1, Automatic}, {1, Automatic}},
PrintingPageRange->{Automatic, Automatic},
PrintingOptions->{"Magnification"->1.,
"PaperOrientation"->"Portrait",
"PaperSize"->{611.25, 789.5625},
"PostScriptOutputFile"->"/users/olness/print.pdf"},
PrivateNotebookOptions->{"VersionedStylesheet"->{"Default.nb"[8.] -> False}},
ShowSelection->True,
Evaluatable->False,
CellHorizontalScrolling->False,
PageBreakWithin->Automatic,
ShowCellLabel->True,
ShowCellTags->False,
AspectRatioFixed->True,
FontFamily->"Times",
FontSize->12,
FrontEndVersion->"11.0 for Linux x86 (64-bit) (July 28, 2016)",
StyleDefinitions->"Default.nb",
RenderingOptions -> {"ObjectDithering" -> True},
RenderingOptions -> {"RasterDithering" -> False}
]
(* End of Notebook Content *)

(* Internal cache information *)
(*CellTagsOutline
CellTagsIndex->{}
*)
(*CellTagsIndex
CellTagsIndex->{}
*)
(*NotebookFileOutline
Notebook[{
Cell[CellGroupData[{
Cell[553, 21, 102, 4, 101, "Section"],
Cell[658, 27, 139, 2, 35, "Input"],
Cell[CellGroupData[{
Cell[822, 33, 157, 3, 51, "Input"],
Cell[982, 38, 72, 1, 30, "Output"]
}, Open  ]],
Cell[1069, 42, 91, 1, 31, "Input"],
Cell[CellGroupData[{
Cell[1185, 47, 210, 3, 31, "Input"],
Cell[1398, 52, 405, 11, 59, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[1840, 68, 168, 2, 31, "Input"],
Cell[2011, 72, 6730, 124, 227, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[8778, 201, 66, 1, 31, "Input"],
Cell[8847, 204, 438, 11, 59, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[9322, 220, 81, 1, 31, "Input"],
Cell[9406, 223, 689, 20, 59, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[10132, 248, 96, 1, 31, "Input"],
Cell[10231, 251, 726, 21, 60, "Output"]
}, Open  ]],
Cell[10972, 275, 148, 2, 31, "Input"],
Cell[CellGroupData[{
Cell[11145, 281, 104, 1, 31, "Input"],
Cell[11252, 284, 319, 6, 32, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[11608, 295, 101, 1, 31, "Input"],
Cell[11712, 298, 317, 6, 32, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[12066, 309, 109, 1, 31, "Input"],
Cell[12178, 312, 336, 7, 32, "Output"]
}, Open  ]],
Cell[CellGroupData[{
Cell[12551, 324, 1104, 39, 541, "Input"],
Cell[13658, 365, 22277, 377, 238, "Output"]
}, Open  ]]
}, Open  ]]
}
]
*)

